<?php

namespace App\Components;

use Illuminate\Support\Manager;
use App\Components\Drivers\FreeIpApi;
use App\Components\Drivers\NullDriver;
use App\Components\Drivers\IpApiDriver;
use App\Components\Drivers\MozApiDriver;
use App\Components\Drivers\OcrSpaceDriver;
use App\Components\Drivers\OpenAiRewriter;
use App\Components\Drivers\DefaultRewriter;
use App\Components\Drivers\DefaultScreenshotDriver;
use App\Components\Drivers\TesseractOCRDriver;
use App\Components\Drivers\WhoisDomainChecker;
use App\Components\Drivers\TinypngApiCompressor;
use App\Components\Drivers\ImageCompressorDriver;

class ToolsManager extends Manager
{
    protected $tool;
    protected $settings;

    public function __construct($tool)
    {
        $this->tool = $tool;
        $this->settings = $tool->settings;
    }

    /**
     * Null Driver throws exception
     *
     * @return NullDriver
     */
    public function createNullDriver()
    {
        return new NullDriver;
    }

    /**
     * Tesseract OCR Driver
     *
     * @return TesseractOCRDriver
     */
    public function createTesseractOCRDriver()
    {
        return new TesseractOCRDriver();
    }

    /**
     * IP-API.com Driver
     *
     * @return IpApiDriver
     */
    public function createIpApiDriver()
    {
        return new IpApiDriver();
    }

    /**
     * TinyPng.com driver Driver
     *
     * @return TinypngApiCompressor
     */
    public function createTinypngApiCompressorDriver()
    {
        return new TinypngApiCompressor($this->settings->tinypng_apikey);
    }

    /**
     * Article rewriter
     *
     * @return DefaultRewriter
     */
    public function createDefaultRewriterDriver()
    {
        return new DefaultRewriter();
    }

    /**
     * Article rewriter
     *
     * @return DefaultRewriter
     */
    public function createDefaultScreenshotDriver()
    {
        return new DefaultScreenshotDriver();
    }

    /**
     * OpenAi Rewriter
     *
     * @return OpenAiRewriter
     */
    public function createOpenAiRewriterDriver()
    {
        return new OpenAiRewriter($this->settings->openai_apikey);
    }

    /**
     * TinyPng.com driver Driver
     *
     * @return ImageCompressorDriver
     */
    public function createImageCompressorDriver()
    {
        return new ImageCompressorDriver();
    }

    /**
     * FreeIPAPI.com Driver
     *
     * @return FreeIpApi
     */
    public function createFreeIpApiDriver()
    {
        return new FreeIpApi();
    }

    /**
     * OCRSpace Driver
     *
     * @return OcrSpaceDriver
     */
    public function createOcrSpaceDriver()
    {
        return new OcrSpaceDriver($this->settings->ocr_space_key);
    }

    /**
     * whois Driver
     *
     * @return WhoISApiDriver
     */
    public function createWhoisDomainCheckerDriver()
    {
        return new WhoisDomainChecker();
    }

    /**
     * Moz Driver
     *
     * @return MozApiDriver
     */
    public function createMozApiDriver()
    {
        return new MozApiDriver();
    }


    /**
     * Get a driver instance.
     *
     * @param  string|null  $name
     * @return mixed
     */
    public function channel($name = null)
    {
        return $this->driver($name);
    }

    /**
     *
     * Get the default SMS driver name.
     *
     * @return string
     */
    public function getDefaultDriver()
    {
        return $this->settings?->driver ?? 'null';
    }
}
