<?php

namespace App\Components\Drivers;

use Illuminate\Support\Str;
use Spatie\Browsershot\Browsershot;
use App\Contracts\ToolDriverInterface;
use Illuminate\Support\Facades\Storage;
use Spatie\Browsershot\Exceptions\CouldNotTakeBrowsershot;
use Symfony\Component\Process\Exception\ProcessFailedException;

class DefaultScreenshotDriver implements ToolDriverInterface
{
    public function parse($request)
    {
        $url = $request->input('url');
        $type = $request->input('type', 'desktop');

        $filename = "screenshot-" . time() . ".png";

        $path = config('artisan.temporary_files_path', 'temp') . '/' . date('m') . '/';
        $disk = config('artisan.public_files_disk', 'public');
        Storage::disk($disk)->makeDirectory($path);

        $filepath = Storage::disk($disk)->path($path . $filename);
        $error = false;
        try {
            $browser = Browsershot::url($url)
                ->setOption('args', ['--disable-web-security'])
                ->waitUntilNetworkIdle(true)
                ->ignoreHttpsErrors()
                ->fullPage();

            if (!empty(setting('screenshot_tool_node_modules_path', null))) {
                $nodeModulesPath = setting('screenshot_tool_node_modules_path');
                $browser->setNodeModulePath($nodeModulesPath);
            }

            if (!empty(setting('screenshot_tool_node_path', null))) {
                $nodePath = setting('screenshot_tool_node_path');
                $browser->setNodeBinary($nodePath);
            }

            if (!empty(setting('screenshot_tool_npm_path', null))) {
                $npmPath = setting('screenshot_tool_npm_path');
                $browser->setNpmBinary($npmPath);
            }

            if (!empty(setting('screenshot_tool_chrome_path', null))) {
                $chromePath = setting('screenshot_tool_chrome_path');
                $browser->setChromePath($chromePath);
            }

            if ($type == 'desktop') {
                $browser->windowSize(1920, 1080);
            }

            if ($type == 'mobile') {
                $browser->userAgent(setting('screenshot_tool_agent', 'Mozilla/5.0 (iPhone; CPU iPhone OS 11_0 like Mac OS X) AppleWebKit/604.1.38 (KHTML, like Gecko) Version/11.0 Mobile/15A372 Safari/604.1'))
                    ->windowSize(375, 812)
                    ->mobile()
                    ->touch();
            }
            $browser->save($filepath);
        } catch (ProcessFailedException $e) {
            // info('Screenshot Generator (ProcessFailedException):', $e->getMessage());
            $error = __('common.somethingWentWrong');
        } catch (CouldNotTakeBrowsershot $e) {
            // info('Screenshot Generator (CouldNotTakeBrowsershot):', $e->getMessage());
            $error = $e->getMessage();
        } catch (\Exception $e) {
            $error = $e->getMessage();
        }

        if ($error) {
            return [false, $error];
        }

        $image = Storage::disk($disk)->url($path . $filename);

        return [true, $image];
    }
}
