import React, {ReactNode, useContext} from 'react';
import clsx from 'clsx';
import {DialogContext} from './dialog-context';
import {IconButton} from '../../buttons/icon-button';
import {CloseIcon} from '../../../icons/material/Close';
import {DialogSize} from './dialog';

interface DialogHeaderProps {
  children: ReactNode;
  className?: string;
  color?: string | null;
  onDismiss?: () => void;
  hideDismissButton?: boolean;
  leftAdornment?: ReactNode;
  rightAdornment?: ReactNode;
  size?: DialogSize;
  padding?: string;
  showDivider?: boolean;
}
export function DialogHeader(props: DialogHeaderProps) {
  const {
    children,
    className,
    color,
    onDismiss,
    leftAdornment,
    rightAdornment,
    hideDismissButton = false,
    size,
    showDivider,
  } = props;
  const {labelId, isDismissable, close} = useContext(DialogContext);

  return (
    <div
      className={clsx(
        className,
        'flex items-center justify-between gap-10 flex-shrink-0',
        showDivider && 'border-b',
        getPadding(props),
        color || 'text-main'
      )}
    >
      {leftAdornment}
      <h3
        id={labelId}
        className={clsx(
          className,
          size === 'xs' ? 'text-xs' : 'text-sm',
          'font-semibold leading-5 opacity-90'
        )}
      >
        {children}
      </h3>
      {rightAdornment}
      {isDismissable && !hideDismissButton && (
        <IconButton
          aria-label="Dismiss"
          onClick={() => {
            if (onDismiss) {
              onDismiss();
            } else {
              close();
            }
          }}
          size={size === 'xs' ? 'xs' : 'sm'}
          className={clsx('text-muted ml-auto', rightAdornment && 'sr-only')}
        >
          <CloseIcon />
        </IconButton>
      )}
    </div>
  );
}

function getPadding({size, padding}: DialogHeaderProps) {
  if (padding) {
    return padding;
  }
  switch (size) {
    case 'xs':
      return 'px-14 py-4';
    case 'sm':
      return 'px-18 py-4';
    default:
      return 'px-24 py-6';
  }
}
