import clsx from 'clsx';
import React, {ComponentPropsWithoutRef, ReactNode} from 'react';
import {CheckIcon} from '../../icons/material/Check';

export interface ListItemBaseProps extends ComponentPropsWithoutRef<'div'> {
  startIcon?: ReactNode;
  endIcon?: ReactNode;
  endSection?: ReactNode;
  description?: ReactNode;
  textLabel?: string;
  capitalizeFirst?: boolean;
  isSelected?: boolean;
  isDisabled?: boolean;
  isActive?: boolean;
  className?: string;
  showCheckmark?: boolean;
}

export const ListItemBase = React.forwardRef<HTMLDivElement, ListItemBaseProps>(
  (props, ref) => {
    let {
      startIcon,
      capitalizeFirst,
      children,
      description,
      endIcon,
      endSection,
      isDisabled,
      isActive,
      isSelected,
      showCheckmark,
      ...domProps
    } = props;

    if (!startIcon && showCheckmark) {
      startIcon = (
        <CheckIcon
          size="sm"
          className={clsx('text-primary', !isSelected && 'invisible')}
        />
      );
    }

    const iconClassName = clsx(
      'icon-sm rounded overflow-hidden flex-shrink-0',
      !isDisabled && 'text-muted'
    );

    return (
      <div
        {...domProps}
        aria-disabled={isDisabled}
        className={itemClassName(props)}
        ref={ref}
      >
        {startIcon && <div className={iconClassName}>{startIcon}</div>}
        <div
          className={clsx(
            'mr-auto',
            capitalizeFirst && 'first-letter:capitalize'
          )}
        >
          {children}
          {description && (
            <div
              className={clsx(
                'text-xs mt-4 whitespace-normal',
                isDisabled ? 'text-disabled' : 'text-muted'
              )}
            >
              {description}
            </div>
          )}
        </div>
        {(endIcon || endSection) && (
          <div className={endIcon ? iconClassName : 'text-muted'}>
            {endIcon || endSection}
          </div>
        )}
      </div>
    );
  }
);

interface Props {
  isSelected?: boolean;
  isDisabled?: boolean;
  isActive?: boolean;
  className?: string;
  showCheckmark?: boolean;
}
function itemClassName({
  className,
  isSelected,
  isActive,
  isDisabled,
  showCheckmark,
}: Props): string {
  let state: string = '';
  if (isDisabled) {
    state = 'text-disabled pointer-events-none';
  } else if (isSelected) {
    if (isActive) {
      state = 'bg-primary/focus';
    } else {
      state = 'bg-primary/selected hover:bg-primary/focus';
    }
  } else if (isActive) {
    state = 'hover:bg-fg-base/15 bg-focus';
  } else {
    state = 'hover:bg-hover';
  }

  return clsx(
    'w-full select-none outline-none cursor-pointer',
    'py-8 text-sm truncate flex items-center gap-10 text-main',
    showCheckmark ? 'px-8' : 'px-20',
    state,
    className
  );
}
